/*
 * Decompiled with CFR 0.152.
 */
package org.apache.shindig.common.crypto;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.security.GeneralSecurityException;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.shindig.common.crypto.BlobCrypter;
import org.apache.shindig.common.crypto.BlobCrypterException;
import org.apache.shindig.common.crypto.BlobExpiredException;
import org.apache.shindig.common.crypto.Crypto;
import org.apache.shindig.common.util.CharsetUtil;
import org.apache.shindig.common.util.TimeSource;

public class BasicBlobCrypter
implements BlobCrypter {
    private static final byte CIPHER_KEY_LABEL = 0;
    private static final byte HMAC_KEY_LABEL = 1;
    public static final String TIMESTAMP_KEY = "t";
    public static final int MASTER_KEY_MIN_LEN = 16;
    private static final long CLOCK_SKEW_ALLOWANCE = 180L;
    private static final String UTF8 = "UTF-8";
    public TimeSource timeSource = new TimeSource();
    private byte[] cipherKey;
    private byte[] hmacKey;

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public BasicBlobCrypter(File keyfile) throws IOException {
        FileInputStream openFile = null;
        try {
            openFile = new FileInputStream(keyfile);
            BufferedReader reader = new BufferedReader(new InputStreamReader((InputStream)openFile, CharsetUtil.UTF8));
            String line = reader.readLine();
            line = line.trim();
            byte[] keyBytes = CharsetUtil.getUtf8Bytes(line);
            this.init(keyBytes);
        }
        finally {
            try {
                if (openFile != null) {
                    openFile.close();
                }
            }
            catch (IOException iOException) {}
        }
    }

    public BasicBlobCrypter(byte[] masterKey) {
        this.init(masterKey);
    }

    private void init(byte[] masterKey) {
        Preconditions.checkArgument((masterKey.length >= 16 ? 1 : 0) != 0, (String)"Master key needs at least %s bytes", (Object[])new Object[]{16});
        this.cipherKey = this.deriveKey((byte)0, masterKey, 16);
        this.hmacKey = this.deriveKey((byte)1, masterKey, 0);
    }

    private byte[] deriveKey(byte label, byte[] masterKey, int len) {
        byte[] base = Crypto.concat(new byte[]{label}, masterKey);
        byte[] hash = DigestUtils.sha((byte[])base);
        if (len == 0) {
            return hash;
        }
        byte[] out = new byte[len];
        System.arraycopy(hash, 0, out, 0, out.length);
        return out;
    }

    @Override
    public String wrap(Map<String, String> in) throws BlobCrypterException {
        Preconditions.checkArgument((!in.containsKey(TIMESTAMP_KEY) ? 1 : 0) != 0, (String)"No '%s' key allowed for BlobCrypter", (Object[])new Object[]{TIMESTAMP_KEY});
        try {
            byte[] encoded = this.serializeAndTimestamp(in);
            byte[] cipherText = Crypto.aes128cbcEncrypt(this.cipherKey, encoded);
            byte[] hmac = Crypto.hmacSha1(this.hmacKey, cipherText);
            byte[] b64 = Base64.encodeBase64((byte[])Crypto.concat(cipherText, hmac));
            return new String(b64, UTF8);
        }
        catch (UnsupportedEncodingException e) {
            throw new BlobCrypterException(e);
        }
        catch (GeneralSecurityException e) {
            throw new BlobCrypterException(e);
        }
    }

    private byte[] serializeAndTimestamp(Map<String, String> in) throws UnsupportedEncodingException {
        StringBuilder sb = new StringBuilder();
        for (Map.Entry<String, String> val : in.entrySet()) {
            sb.append(URLEncoder.encode(val.getKey(), UTF8));
            sb.append('=');
            sb.append(URLEncoder.encode(val.getValue(), UTF8));
            sb.append('&');
        }
        sb.append(TIMESTAMP_KEY);
        sb.append('=');
        sb.append(this.timeSource.currentTimeMillis() / 1000L);
        return sb.toString().getBytes(UTF8);
    }

    @Override
    public Map<String, String> unwrap(String in, int maxAgeSec) throws BlobCrypterException {
        try {
            byte[] bin = Base64.decodeBase64((byte[])in.getBytes());
            byte[] hmac = new byte[20];
            byte[] cipherText = new byte[bin.length - 20];
            System.arraycopy(bin, 0, cipherText, 0, cipherText.length);
            System.arraycopy(bin, cipherText.length, hmac, 0, hmac.length);
            Crypto.hmacSha1Verify(this.hmacKey, cipherText, hmac);
            byte[] plain = Crypto.aes128cbcDecrypt(this.cipherKey, cipherText);
            Map<String, String> out = this.deserialize(plain);
            this.checkTimestamp(out, maxAgeSec);
            return out;
        }
        catch (GeneralSecurityException e) {
            throw new BlobCrypterException("Invalid token signature", e);
        }
        catch (ArrayIndexOutOfBoundsException e) {
            throw new BlobCrypterException("Invalid token format", e);
        }
        catch (NegativeArraySizeException e) {
            throw new BlobCrypterException("Invalid token format", e);
        }
        catch (UnsupportedEncodingException e) {
            throw new BlobCrypterException(e);
        }
    }

    private Map<String, String> deserialize(byte[] plain) throws UnsupportedEncodingException {
        String base = new String(plain, UTF8);
        String[] items = base.split("[&=]");
        HashMap map = Maps.newHashMapWithExpectedSize((int)items.length);
        int i = 0;
        while (i < items.length) {
            String key = URLDecoder.decode(items[i++], UTF8);
            String val = URLDecoder.decode(items[i++], UTF8);
            map.put(key, val);
        }
        return map;
    }

    private void checkTimestamp(Map<String, String> out, int maxAge) throws BlobExpiredException {
        long origin = Long.parseLong(out.get(TIMESTAMP_KEY));
        long minTime = origin - 180L;
        long maxTime = origin + (long)maxAge + 180L;
        long now = this.timeSource.currentTimeMillis() / 1000L;
        if (minTime >= now || now >= maxTime) {
            throw new BlobExpiredException(minTime, now, maxTime);
        }
    }
}

