/*
 * Decompiled with CFR 0.152.
 */
package org.apache.shindig.gadgets.rewrite.image;

import com.google.common.collect.ImmutableSet;
import com.google.inject.Inject;
import java.awt.AlphaComposite;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.InputStream;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.imageio.ImageIO;
import javax.imageio.ImageWriter;
import org.apache.sanselan.ImageFormat;
import org.apache.sanselan.ImageInfo;
import org.apache.sanselan.ImageReadException;
import org.apache.sanselan.Sanselan;
import org.apache.sanselan.common.byteSources.ByteSource;
import org.apache.sanselan.common.byteSources.ByteSourceInputStream;
import org.apache.shindig.common.uri.Uri;
import org.apache.shindig.gadgets.http.HttpRequest;
import org.apache.shindig.gadgets.http.HttpResponse;
import org.apache.shindig.gadgets.http.HttpResponseBuilder;
import org.apache.shindig.gadgets.rewrite.image.BMPOptimizer;
import org.apache.shindig.gadgets.rewrite.image.BaseOptimizer;
import org.apache.shindig.gadgets.rewrite.image.GIFOptimizer;
import org.apache.shindig.gadgets.rewrite.image.ImageRewriter;
import org.apache.shindig.gadgets.rewrite.image.ImageUtils;
import org.apache.shindig.gadgets.rewrite.image.JPEGOptimizer;
import org.apache.shindig.gadgets.rewrite.image.OptimizerConfig;
import org.apache.shindig.gadgets.rewrite.image.PNGOptimizer;

public class BasicImageRewriter
implements ImageRewriter {
    private static final String CONTENT_TYPE_IMAGE_PNG = "image/png";
    private static final String RESIZE_IMAGE_TOO_LARGE = "The image is too large to resize";
    private static final String RESIZE_OUTPUT_FORMAT = "png";
    private static final String CONTENT_LENGTH = "Content-Length";
    public static final String PARAM_RESIZE_QUALITY = "resize_q";
    public static final String PARAM_RESIZE_WIDTH = "resize_w";
    public static final String PARAM_RESIZE_HEIGHT = "resize_h";
    public static final String PARAM_NO_EXPAND = "no_expand";
    public static final String CONTENT_TYPE_AND_EXTENSION_MISMATCH = "Content is not an image but file extension asserts it is";
    public static final String CONTENT_TYPE_AND_MIME_MISMATCH = "Content is not an image but mime type asserts it is";
    private static final int DEFAULT_QUALITY = 100;
    private static final int BITS_PER_BYTE = 8;
    private static final Color COLOR_TRANSPARENT = new Color(255, 255, 255, 0);
    private static final String CONTENT_TYPE = "Content-Type";
    private static final Logger log = Logger.getLogger(BasicImageRewriter.class.getName());
    private static final Set<String> SUPPORTED_MIME_TYPES = ImmutableSet.of((Object)"image/gif", (Object)"image/png", (Object)"image/jpeg", (Object)"image/bmp");
    private static final Set<String> SUPPORTED_FILE_EXTENSIONS = ImmutableSet.of((Object)".gif", (Object)".png", (Object)".jpeg", (Object)".jpg", (Object)".bmp");
    private final OptimizerConfig config;
    private final AtomicLong totalSourceImageSize = new AtomicLong();
    private final AtomicLong totalRewrittenImageBytes = new AtomicLong();

    @Inject
    public BasicImageRewriter(OptimizerConfig config) {
        this.config = config;
    }

    @Override
    public HttpResponse rewrite(HttpRequest request, HttpResponse response) {
        Uri uri = request.getUri();
        if (uri == null || request == null || response == null) {
            return response;
        }
        try {
            boolean isResizeRequested;
            Integer resizeQuality = request.getParamAsInteger(PARAM_RESIZE_QUALITY);
            Integer requestedWidth = request.getParamAsInteger(PARAM_RESIZE_WIDTH);
            Integer requestedHeight = request.getParamAsInteger(PARAM_RESIZE_HEIGHT);
            boolean bl = isResizeRequested = requestedWidth != null || requestedHeight != null;
            if (!this.isSupportedContent(response) && !this.isImage(uri)) {
                return response;
            }
            if (!(this.isUsableParameter(requestedWidth) && this.isUsableParameter(requestedHeight) && this.isUsableParameter(resizeQuality))) {
                return response;
            }
            ImageFormat imageFormat = Sanselan.guessFormat((ByteSource)new ByteSourceInputStream(response.getResponse(), uri.getPath()));
            if (imageFormat == ImageFormat.IMAGE_FORMAT_UNKNOWN) {
                return this.enforceUnreadableImageRestrictions(uri, response);
            }
            if (response.getContentLength() < this.config.getMinThresholdBytes()) {
                return response;
            }
            ImageInfo imageInfo = Sanselan.getImageInfo((InputStream)response.getResponse(), (String)uri.getPath());
            boolean noExpand = "1".equals(request.getParam(PARAM_NO_EXPAND));
            if (noExpand && imageInfo.getHeight() <= requestedHeight && imageInfo.getWidth() <= requestedWidth) {
                isResizeRequested = false;
            }
            boolean isOversizedImage = this.isImageTooLarge(imageInfo);
            if (isResizeRequested && isOversizedImage) {
                HttpResponseBuilder rejectedResponseBuilder = new HttpResponseBuilder().setHttpStatusCode(403).setResponseString(RESIZE_IMAGE_TOO_LARGE);
                return rejectedResponseBuilder.create();
            }
            if (imageInfo.getNumberOfImages() > 1 || isOversizedImage) {
                return response;
            }
            int originalContentSize = response.getContentLength();
            this.totalSourceImageSize.addAndGet(originalContentSize);
            BufferedImage image = this.readImage(imageFormat, response);
            if (isResizeRequested) {
                int origWidth = imageInfo.getWidth();
                int origHeight = imageInfo.getHeight();
                int widthDelta = 0;
                int heightDelta = 0;
                if (requestedWidth == null || requestedHeight == null) {
                    if (requestedWidth == null) {
                        requestedWidth = Math.max(1, (int)((double)origWidth / (double)origHeight * (double)requestedHeight.intValue()));
                    }
                    if (requestedHeight == null) {
                        requestedHeight = Math.max(1, (int)((double)origHeight / (double)origWidth * (double)requestedWidth.intValue()));
                    }
                } else {
                    double ratio = this.getResizeRatio(requestedWidth, requestedHeight, origWidth, origHeight);
                    int widthAfterStep1 = Math.max(1, (int)(ratio * (double)origWidth));
                    widthDelta = requestedWidth - widthAfterStep1;
                    int heightAfterStep1 = Math.max(1, (int)(ratio * (double)origHeight));
                    heightDelta = requestedHeight - heightAfterStep1;
                    if (noExpand) {
                        if (widthDelta == 0) {
                            requestedHeight = heightAfterStep1;
                            heightDelta = 0;
                        } else if (heightDelta == 0) {
                            requestedWidth = widthAfterStep1;
                            widthDelta = 0;
                        }
                    }
                }
                if (resizeQuality == null) {
                    resizeQuality = 100;
                }
                if (this.isResizeRequired(requestedWidth, requestedHeight, imageInfo) && !this.isTargetImageTooLarge(requestedWidth, requestedHeight, imageInfo)) {
                    image = this.resizeImage(image, requestedWidth, requestedHeight, widthDelta, heightDelta);
                    response = this.updateResponse(response, image);
                }
            }
            response = this.getOptimizer(response, imageFormat, image);
            this.totalRewrittenImageBytes.addAndGet(response.getContentLength());
        }
        catch (IOException ioe) {
            log.log(Level.WARNING, "IO Error rewriting image " + request.toString() + " - " + ioe.getMessage());
        }
        catch (RuntimeException re) {
            log.log(Level.INFO, "Unknown error rewriting image " + request.toString(), re);
        }
        catch (ImageReadException ire) {
            log.log(Level.INFO, "Failed to read image. Skipping " + request.toString(), ire.getMessage());
        }
        return response;
    }

    private HttpResponse updateResponse(HttpResponse response, BufferedImage image) throws IOException {
        ImageWriter imageWriter = ImageIO.getImageWritersByFormatName(RESIZE_OUTPUT_FORMAT).next();
        BaseOptimizer.ImageIOOutputter outputter = new BaseOptimizer.ImageIOOutputter(imageWriter, null);
        byte[] imageBytes = outputter.toBytes(image);
        HttpResponseBuilder newResponseBuilder = new HttpResponseBuilder(response).setResponse(imageBytes).setHeader(CONTENT_TYPE, CONTENT_TYPE_IMAGE_PNG).setHeader(CONTENT_LENGTH, String.valueOf(imageBytes.length));
        return newResponseBuilder.create();
    }

    private boolean isUsableParameter(Integer parameterValue) {
        if (parameterValue == null) {
            return true;
        }
        return parameterValue > 0;
    }

    private double getResizeRatio(int requestedWidth, int requestedHeight, int origWidth, int origHeight) {
        double ratio = Math.min((double)requestedWidth / (double)origWidth, (double)requestedHeight / (double)origHeight);
        return ratio;
    }

    private BufferedImage resizeImage(BufferedImage image, Integer requestedWidth, Integer requestedHeight, int extraWidth, int extraHeight) {
        int imageType;
        int heightStretch;
        int widthStretch = requestedWidth - extraWidth;
        if ((image = ImageUtils.getScaledInstance(image, widthStretch, heightStretch = requestedHeight - extraHeight, RenderingHints.VALUE_INTERPOLATION_BICUBIC, true, imageType = ImageUtils.isOpaque(image) ? 5 : 2)).getWidth() != requestedWidth.intValue() || image.getHeight() != requestedHeight.intValue()) {
            image = this.stretchImage(image, requestedWidth, requestedHeight, imageType);
        }
        return image;
    }

    private BufferedImage stretchImage(BufferedImage image, Integer requestedWidth, Integer requestedHeight, int imageType) {
        BufferedImage scaledImage = new BufferedImage(requestedWidth, requestedHeight, imageType);
        Graphics2D g2d = scaledImage.createGraphics();
        g2d.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BICUBIC);
        this.fillWithTransparent(g2d, requestedWidth, requestedHeight);
        g2d.drawImage(image, 0, 0, requestedWidth, requestedHeight, null);
        image = scaledImage;
        return image;
    }

    private void fillWithTransparent(Graphics2D g2d, Integer requestedWidth, Integer requestedHeight) {
        g2d.setComposite(AlphaComposite.Clear);
        g2d.setColor(COLOR_TRANSPARENT);
        g2d.fillRect(0, 0, requestedWidth, requestedHeight);
        g2d.setComposite(AlphaComposite.SrcOver);
    }

    private HttpResponse getOptimizer(HttpResponse response, ImageFormat imageFormat, BufferedImage image) throws IOException {
        if (imageFormat == ImageFormat.IMAGE_FORMAT_GIF) {
            if (!response.getResponseAsString().contains("NETSCAPE2.0")) {
                response = new GIFOptimizer(this.config, response).rewrite(image);
            }
        } else if (imageFormat == ImageFormat.IMAGE_FORMAT_PNG) {
            response = new PNGOptimizer(this.config, response).rewrite(image);
        } else if (imageFormat == ImageFormat.IMAGE_FORMAT_JPEG) {
            response = new JPEGOptimizer(this.config, response).rewrite(image);
        } else if (imageFormat == ImageFormat.IMAGE_FORMAT_BMP) {
            response = new BMPOptimizer(this.config, response).rewrite(image);
        }
        return response;
    }

    private boolean isImageTooLarge(ImageInfo imageInfo) {
        return this.isTargetImageTooLarge(imageInfo.getWidth(), imageInfo.getHeight(), imageInfo);
    }

    private boolean isTargetImageTooLarge(int requestedHeight, int requestedWidth, ImageInfo imageInfo) {
        long imagePixels = Math.abs(requestedHeight) * Math.abs(requestedWidth);
        long imageSizeBits = imagePixels * (long)imageInfo.getBitsPerPixel();
        return imageSizeBits > (long)(this.config.getMaxInMemoryBytes() * 8);
    }

    private boolean isSupportedContent(HttpResponse response) {
        return SUPPORTED_MIME_TYPES.contains(response.getHeader(CONTENT_TYPE));
    }

    private boolean isImage(Uri uri) {
        boolean pathExtMatches = false;
        for (String ext : SUPPORTED_FILE_EXTENSIONS) {
            if (!uri.getPath().endsWith(ext)) continue;
            pathExtMatches = true;
            break;
        }
        return pathExtMatches;
    }

    private boolean isResizeRequired(int resize_w, int resize_h, ImageInfo imageInfo) {
        return resize_w != imageInfo.getWidth() || resize_h != imageInfo.getHeight();
    }

    HttpResponse enforceUnreadableImageRestrictions(Uri uri, HttpResponse original) {
        String contentType = original.getHeader(CONTENT_TYPE);
        if (contentType != null) {
            contentType = contentType.toLowerCase();
            for (String expected : SUPPORTED_MIME_TYPES) {
                if (!contentType.contains(expected)) continue;
                return new HttpResponseBuilder(original).setHttpStatusCode(415).setResponseString(CONTENT_TYPE_AND_MIME_MISMATCH).create();
            }
        }
        String path = uri.getPath().toLowerCase();
        for (String supportedExtension : SUPPORTED_FILE_EXTENSIONS) {
            if (!path.endsWith(supportedExtension)) continue;
            return new HttpResponseBuilder(original).setHttpStatusCode(415).setResponseString(CONTENT_TYPE_AND_EXTENSION_MISMATCH).create();
        }
        return original;
    }

    @Override
    public long getOriginalImageBytes() {
        return this.totalSourceImageSize.get();
    }

    @Override
    public long getRewrittenImageBytes() {
        return this.totalRewrittenImageBytes.get();
    }

    protected BufferedImage readImage(ImageFormat imageFormat, HttpResponse response) throws ImageReadException, IOException {
        if (imageFormat == ImageFormat.IMAGE_FORMAT_GIF) {
            return this.readGif(response);
        }
        if (imageFormat == ImageFormat.IMAGE_FORMAT_PNG) {
            return this.readPng(response);
        }
        if (imageFormat == ImageFormat.IMAGE_FORMAT_JPEG) {
            return this.readJpeg(response);
        }
        if (imageFormat == ImageFormat.IMAGE_FORMAT_BMP) {
            return this.readBmp(response);
        }
        throw new ImageReadException("Unsupported format " + imageFormat.name);
    }

    protected BufferedImage readBmp(HttpResponse response) throws ImageReadException, IOException {
        return BMPOptimizer.readBmp(response.getResponse());
    }

    protected BufferedImage readPng(HttpResponse response) throws ImageReadException, IOException {
        return PNGOptimizer.readPng(response.getResponse());
    }

    protected BufferedImage readGif(HttpResponse response) throws ImageReadException, IOException {
        return GIFOptimizer.readGif(response.getResponse());
    }

    protected BufferedImage readJpeg(HttpResponse response) throws ImageReadException, IOException {
        return JPEGOptimizer.readJpeg(response.getResponse());
    }
}

