/* Search for a Settings document, returns it. If not found, creates one.
   WARNING: There should be only one
   
   Configuration parameters can be used to set:
   # Where to create the document. Root by default ("/")
   ast.settings.parent.path=/
   # Default values. This is JSON string, must respect field names,
   # and remember settings:settings is a multivalued complex field
   # and all values are strings (except blob, can't be set in nuxeo.conf)
   # typeOfData must be a valid directory entry
   ast.settings.defaultValues=
   
   Example of default values set in nuxeo.conf:
   ast.settings.defaultValues=[{"module": "AI", "setting": "runAI", "value": "true", "typeOfData": "boolean"},{"module": "AI", "setting": "minConfidence", "value": "0.8", "typeOfData": "double", "description": "Tags will not be saved if confidence is less than 80%"}]
   
   By default, the Settings document is set to be Read only for "members" and Everything fo administrators
*/
function run(input, params) {
  
  var docs, settingsDoc, parentPath, parentDoc, defaultValues;
  
  //Console.log("astSettings_GetDocCreateIfNeeded...");
  
  docs = Repository.Query(
    null, {
      "query": "SELECT * FROM Document WHERE ecm:primaryType = 'astSettings' AND ecm:isVersion = 0 AND ecm:isProxy = 0"
    });

  if(docs.size() === 0) {
    parentPath = Env["ast.settings.parent.path"];
    if(!parentPath) {
      parentPath = "/";
    }
    defaultValues = Env["ast.settings.defaultValues"];
    if(defaultValues) {
      defaultValues = JSON.parse(defaultValues);
    } else {
      defaultValues = null;
    }
    
    Console.log("astSettings_GetDocCreateIfNeeded: Settings document not found => creating it. Configuration parameters:\nparentPath: " + parentPath + "\ndefaultValues:\n" + JSON.stringify(defaultValues, null, 2));
    
    Auth.LoginAs(null, {});
    parentDoc = Repository.GetDocument(null, {'value': parentPath});
    settingsDoc = Document.Create(
      parentDoc, {
        'type': "astSettings",
        'name': "Settings",
        'properties': {
          "dc:title": "Settings",
          "astsettings:settings": defaultValues
        }
      });
    // ==========================================================
    // 2022-08. Strangely enough, the title is not set after this Document.Create call...
    settingsDoc = Document.SetProperty(
      settingsDoc, {
        'xpath': "dc:title",
        'save': true,
        'value': "Settings"
      });
    // ==========================================================
    
    settingsDoc = Document.BlockPermissionInheritance(settingsDoc, {});
    settingsDoc = Document.AddACE(
      settingsDoc, {
        'permission': "Read",
        'user': "members",
      });
    settingsDoc = Document.AddACE(
      settingsDoc, {
        'permission': "Everything",
        'user': "administrators",
      });
    Auth.Logout(null, {});
    settingsDoc.doc.refresh();
    
    Session.save();
    org.nuxeo.runtime.transaction.TransactionHelper.commitOrRollbackTransaction();
    org.nuxeo.runtime.transaction.TransactionHelper.startTransaction();
    
  } else {
    settingsDoc = docs[0];
    if(docs.size() > 1) {
      Console.log("Warning: There should be only one astSettings document, there are " + docs.size() + ".");
    }
  }
  
  return settingsDoc;

}